/*:
 * @target MZ
 * @plugindesc 立ち絵管理プラグイン - 衣装と表情の切り替え対応
 * @command setCharacterState
 * @text 立ち絵状態を設定
 * @desc 衣装と表情を設定します。
 *
 * @arg characterId
 * @text キャラクターID
 * @type string
 * @desc 操作対象のキャラクターID。
 *
 * @arg expression
 * @text 表情
 * @type select
 * @option 喜1
 * @value happy1
 * @option 喜2
 * @value happy2
 * @option 怒1
 * @value angry1
 * @option 怒2
 * @value angry2
 * @option 哀1
 * @value sad1
 * @option 哀2
 * @value sad2
 * @option 楽1
 * @value fun1
 * @option 楽2
 * @value fun2
 * @desc キャラクターの表情を選択。
 *
 * @arg bra
 * @text ブラ
 * @type boolean
 * @on 着用
 * @off 脱衣
 * @desc ブラの着用状態を設定します。
 *
 * @arg underwear
 * @text 下着
 * @type boolean
 * @on 着用
 * @off 脱衣
 * @desc 下着の着用状態を設定します。
 *
 * @arg mask
 * @text マスク
 * @type boolean
 * @on 着用
 * @off 脱衣
 * @desc マスクの着用状態を設定します。
 *
 * @arg x
 * @text 表示位置X
 * @type number
 * @min -Infinity
 * @max Infinity
 * @default null
 * @desc 特殊な表示位置X座標を指定します。デフォルト位置を使用する場合は空白。
 *
 * @arg y
 * @text 表示位置Y
 * @type number
 * @min -Infinity
 * @max Infinity
 * @default null
 * @desc 特殊な表示位置Y座標を指定します。デフォルト位置を使用する場合は空白。
 *
 * @command hideCharacter
 * @text 立ち絵非表示
 * @desc 指定したキャラクターの立ち絵をフェードアウトして非表示にします。
 *
 * @arg characterId
 * @text キャラクターID
 * @type string
 * @desc 非表示にするキャラクターID。
 *
 * @command disableCharacter
 * @text 立ち絵無効化
 * @desc 指定したキャラクターの立ち絵を無効化します。
 *
 * @arg characterId
 * @text キャラクターID
 * @type string
 * @desc 無効化するキャラクターID。
 */

(() => {
    const pluginName = "CharacterStateManager";
    const defaultX = 0;
    const defaultY = 0;

    PluginManager.registerCommand(pluginName, "setCharacterState", args => {
        const characterId = args.characterId;
        const expression = args.expression;
        const bra = args.bra === "true";
        const underwear = args.underwear === "true";
        const mask = args.mask === "true";
        const x = args.x ? Number(args.x) : defaultX;
        const y = args.y ? Number(args.y) : defaultY;

        $gameSystem.setCharacterState(characterId, {
            expression,
            bra,
            underwear,
            mask,
            x,
            y
        });
    });

    PluginManager.registerCommand(pluginName, "hideCharacter", args => {
        const characterId = args.characterId;
        $gameSystem.hideCharacterSprite(characterId);
    });

    PluginManager.registerCommand(pluginName, "disableCharacter", args => {
        const characterId = args.characterId;
        $gameSystem.disableCharacterSprite(characterId);
    });

    Game_System.prototype.initialize = function() {
        this._characterStates = {};
    };

    Game_System.prototype.setCharacterState = function(characterId, state) {
        if (!this._characterStates) {
            this._characterStates = {};
        }
        this._characterStates[characterId] = state;
        this.refreshCharacterSprite(characterId);
    };

    Game_System.prototype.refreshCharacterSprite = function(characterId) {
        const state = this._characterStates[characterId];
        if (!state) return;

        const basePath = `img/pictures/characters/`;
        const braState = state.bra ? "bra_on" : "bra_off";
        const underwearState = state.underwear ? "underwear_on" : "underwear_off";
        const maskState = state.mask ? "mask_on" : "mask_off";

        const fileName = `${characterId}_${state.expression}_${braState}_${underwearState}_${maskState}.png`;

        const pictureId = 1; // 固定値、もしくは動的に設定可能
        $gameScreen.showPicture(pictureId, basePath + fileName, 0, state.x, state.y, 100, 100, 255, 0);
    };

    Game_System.prototype.hideCharacterSprite = function(characterId) {
        const pictureId = 1; // 固定値、もしくは動的に設定可能
        $gameScreen.erasePicture(pictureId);
    };

    Game_System.prototype.disableCharacterSprite = function(characterId) {
        if (this._characterStates && this._characterStates[characterId]) {
            delete this._characterStates[characterId];
        }
        this.hideCharacterSprite(characterId);
    };
})();
